<?php
/* --------------------------------------------------------------
 LanguageFinder.php 2020-02-25
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Application\Kernel\Bootstrapper\Language;

use Doctrine\DBAL\Connection;
use Gambio\Core\Configuration\ConfigurationService;
use UnexpectedValueException;

/**
 * Class LanguageFinder
 * @package Gambio\Admin\Application\Middleware\Language
 *
 * @Todo    : Maybe change path/namespace from LanguageFinder and Language
 */
class LanguageFinder
{
    /**
     * @var ConfigurationService
     */
    private $configurationService;
    
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * LanguageFinder constructor.
     *
     * @param ConfigurationService $configurationService
     * @param Connection           $connection
     */
    public function __construct(ConfigurationService $configurationService, Connection $connection)
    {
        $this->configurationService = $configurationService;
        $this->connection           = $connection;
    }
    
    
    /**
     * Loads active language.
     *
     * @param string|null $newLanguage
     *
     * @return Language
     */
    public function load(?string $newLanguage): Language
    {
        $languageData = $this->findActiveLanguage($newLanguage);
        
        return Language::fromArray($languageData);
    }
    
    
    /**
     * Finds data for active language.
     *
     * @param string|null $newLanguage
     *
     * @return array
     */
    private function findActiveLanguage(?string $newLanguage): array
    {
        $query      = 'SELECT * FROM `languages` WHERE `status_admin` = "1" ORDER BY `sort_order`;';
        $languages  = $this->connection->fetchAll($query);
        $useDefault = false;
        
        if (!$newLanguage) {
            $defaultLanguage = $this->configurationService->find('configuration/DEFAULT_LANGUAGE');
            $newLanguage     = $defaultLanguage->value();
            $useDefault      = true;
        }
        
        foreach ($languages as $key => $language) {
            if ($language['code'] === $newLanguage) {
                return $language;
            }
        }
        
        $msg = "Language not found ({$newLanguage}).";
        if ($useDefault) {
            $msg = "(Default) " . lcfirst($msg);
        }
        
        throw new UnexpectedValueException($msg);
    }
}